<?php
// 核心配置（可自定义）
define('SCAN_DIR', __DIR__); // 扫描的目标文件夹（当前目录）
define('EXCLUDE_FILES', ['index.php', '.htaccess']); // 禁止显示/下载的文件
define('ALLOW_HIDDEN', false); // 是否显示隐藏文件（以.开头）

// 定义文件分类规则（可根据后缀自定义）
$fileCategories = [
    '文档类' => ['doc', 'docx', 'pdf', 'txt', 'ppt', 'pptx', 'xls', 'xlsx', 'md'],
    '压缩包' => ['zip', 'rar', '7z', 'tar', 'gz'],
    '图片类' => ['jpg', 'jpeg', 'png', 'gif', 'webp', 'svg'],
    '视频类' => ['mp4', 'avi', 'mov', 'mkv', 'flv'],
    '音频类' => ['mp3', 'wav', 'flac', 'm4a'],
    '其他类' => [] // 未匹配到的文件自动归为「其他类」
];

// 扫描文件夹，筛选并分类文件
$files = scandir(SCAN_DIR);
$allFiles = []; // 所有合法文件
$categoryFiles = []; // 按分类存储的文件

// 初始化分类数组
foreach (array_keys($fileCategories) as $category) {
    $categoryFiles[$category] = [];
}

// 筛选文件并分类
foreach ($files as $file) {
    $filePath = SCAN_DIR . DIRECTORY_SEPARATOR . $file;
    // 筛选条件：是文件 + 不在禁止列表 + 非隐藏文件（可选）
    $isValid = is_file($filePath) 
        && !in_array($file, EXCLUDE_FILES)
        && (ALLOW_HIDDEN || substr($file, 0, 1) !== '.');
    
    if ($isValid) {
        // 获取文件后缀（小写）
        $fileExt = strtolower(pathinfo($file, PATHINFO_EXTENSION));
        // 匹配分类
        $category = '其他类';
        foreach ($fileCategories as $catName => $extList) {
            if (in_array($fileExt, $extList)) {
                $category = $catName;
                break;
            }
        }
        // 存储文件信息（名称、大小、修改时间、分类、后缀）
        $fileInfo = [
            'name' => $file,
            'size' => filesize($filePath),
            'mtime' => filemtime($filePath),
            'category' => $category,
            'ext' => $fileExt
        ];
        $allFiles[] = $fileInfo;
        $categoryFiles[$category][] = $fileInfo;
    }
}

// 文件大小格式化（B→KB→MB→GB）
function formatFileSize($bytes) {
    if ($bytes < 1024) return $bytes . ' B';
    elseif ($bytes < 1048576) return round($bytes / 1024, 2) . ' KB';
    elseif ($bytes < 1073741824) return round($bytes / 1048576, 2) . ' MB';
    else return round($bytes / 1073741824, 2) . ' GB';
}

// 时间格式化（修改时间）
function formatTime($timestamp) {
    return date('Y-m-d H:i:s', $timestamp);
}

// 下载功能逻辑（安全校验）
if (isset($_GET['download'])) {
    $downloadName = urldecode($_GET['download']);
    $downloadPath = SCAN_DIR . DIRECTORY_SEPARATOR . $downloadName;
    
    // 双重安全校验：文件存在 + 是合法文件 + 不在禁止列表
    $isSafe = file_exists($downloadPath) 
        && is_file($downloadPath)
        && !in_array($downloadName, EXCLUDE_FILES)
        && (ALLOW_HIDDEN || substr($downloadName, 0, 1) !== '.');
    
    if ($isSafe) {
        // 触发浏览器下载（支持中文/特殊字符文件名）
        header('Content-Description: File Transfer');
        header('Content-Type: application/octet-stream');
        header('Content-Disposition: attachment; filename="' . rawurlencode($downloadName) . '"');
        header('Expires: 0');
        header('Cache-Control: must-revalidate, post-check=0, pre-check=0');
        header('Pragma: public');
        header('Content-Length: ' . filesize($downloadPath));
        ob_clean();
        flush();
        readfile($downloadPath);
        exit;
    } else {
        echo '<script>alert("文件不存在或无下载权限！");window.location.href="index.php";</script>';
        exit;
    }
}

// 分类筛选逻辑
$activeCategory = isset($_GET['category']) ? $_GET['category'] : '全部';
$displayFiles = $activeCategory === '全部' ? $allFiles : (isset($categoryFiles[$activeCategory]) ? $categoryFiles[$activeCategory] : []);
?>
<!DOCTYPE html>
<html lang="zh-CN">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>文件下载中心</title>
    <style>
        /* 基础样式重置 */
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
            font-family: 'Inter', '微软雅黑', sans-serif;
        }
        body {
            background-color: #f0f2f5;
            line-height: 1.6;
            color: #2d3748;
            min-height: 100vh;
            display: flex;
            flex-direction: column;
        }
        /* 顶部导航栏 */
        .navbar {
            background-color: #fff;
            box-shadow: 0 2px 10px rgba(0, 0, 0, 0.05);
            padding: 0 30px;
            position: sticky;
            top: 0;
            z-index: 999;
        }
        .nav-container {
            max-width: 1200px;
            margin: 0 auto;
            display: flex;
            justify-content: space-between;
            align-items: center;
            height: 70px;
        }
        .logo {
            font-size: 22px;
            font-weight: 600;
            color: #4361ee;
            text-decoration: none;
        }
        .nav-menu {
            display: flex;
            list-style: none;
        }
        .nav-item {
            margin-left: 30px;
        }
        .nav-link {
            color: #666;
            text-decoration: none;
            font-size: 15px;
            font-weight: 500;
            transition: color 0.2s ease;
        }
        .nav-link:hover, .nav-link.active {
            color: #4361ee;
        }
        /* 移动端菜单按钮 */
        .menu-btn {
            display: none;
            font-size: 24px;
            color: #4361ee;
            background: none;
            border: none;
            cursor: pointer;
        }

        /* 主体内容区 */
        .main {
            flex: 1;
            padding: 30px 20px;
        }
        .container {
            max-width: 1200px;
            margin: 0 auto;
            background: #fff;
            border-radius: 12px;
            box-shadow: 0 4px 20px rgba(0, 0, 0, 0.05);
            overflow: hidden;
        }
        /* 分类筛选栏 */
        .category-bar {
            padding: 20px 30px;
            border-bottom: 1px solid #eee;
            background-color: #f8f9fa;
        }
        .category-title {
            font-size: 16px;
            font-weight: 600;
            margin-bottom: 15px;
            color: #444;
        }
        .category-list {
            display: flex;
            flex-wrap: wrap;
            gap: 10px;
        }
        .category-item {
            padding: 6px 18px;
            background-color: #fff;
            border: 1px solid #eee;
            border-radius: 20px;
            color: #666;
            text-decoration: none;
            font-size: 14px;
            transition: all 0.2s ease;
            cursor: pointer;
        }
        .category-item:hover, .category-item.active {
            background-color: #4361ee;
            color: #fff;
            border-color: #4361ee;
        }
        /* 文件列表区域 */
        .file-section {
            padding: 30px;
        }
        .section-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 20px;
        }
        .section-title {
            font-size: 20px;
            font-weight: 600;
            color: #333;
        }
        .file-count {
            color: #718096;
            font-size: 14px;
        }
        /* 列表头部（表头） */
        .list-header {
            display: grid;
            grid-template-columns: 5fr 2fr 2fr 1fr;
            gap: 10px;
            padding: 15px 20px;
            background-color: #f8f9fa;
            border-radius: 8px 8px 0 0;
            font-weight: 600;
            color: #666;
            font-size: 14px;
        }
        /* 文件列表 */
        .file-list {
            list-style: none;
            border: 1px solid #eee;
            border-top: none;
            border-radius: 0 0 8px 8px;
        }
        .file-item {
            display: grid;
            grid-template-columns: 5fr 2fr 2fr 1fr;
            gap: 10px;
            padding: 18px 20px;
            align-items: center;
            border-bottom: 1px solid #eee;
            transition: background-color 0.2s ease;
        }
        .file-item:last-child {
            border-bottom: none;
        }
        .file-item:hover {
            background-color: #fafafa;
        }
        /* 文件信息样式 */
        .file-name {
            font-size: 16px;
            white-space: nowrap;
            overflow: hidden;
            text-overflow: ellipsis;
            display: flex;
            align-items: center;
        }
        .file-name i {
            margin-right: 8px;
            color: #9f7aea;
        }
        .file-meta {
            color: #718096;
            font-size: 14px;
        }
        /* 下载按钮 */
        .download-btn {
            padding: 8px 16px;
            background-color: #4361ee;
            color: #fff;
            border: none;
            border-radius: 6px;
            cursor: pointer;
            font-size: 14px;
            font-weight: 500;
            transition: background-color 0.2s ease;
            text-align: center;
            text-decoration: none;
        }
        .download-btn:hover {
            background-color: #3a0ca3;
        }
        /* 无文件提示 */
        .no-files {
            text-align: center;
            padding: 80px 20px;
            color: #9f7aea;
            font-size: 18px;
        }
        .no-files i {
            font-size: 48px;
            margin-bottom: 20px;
            display: block;
            opacity: 0.5;
        }

        /* 底部导航栏（页脚） */
        .footer {
            background-color: #2d3748;
            color: #fff;
            padding: 40px 30px 20px;
            margin-top: auto;
        }
        .footer-container {
            max-width: 1200px;
            margin: 0 auto;
            display: grid;
            grid-template-columns: repeat(3, 1fr);
            gap: 40px;
            margin-bottom: 30px;
        }
        .footer-col h3 {
            font-size: 18px;
            font-weight: 600;
            margin-bottom: 20px;
            color: #4361ee;
        }
        .footer-link {
            display: block;
            color: #ddd;
            text-decoration: none;
            font-size: 14px;
            margin-bottom: 10px;
            transition: color 0.2s ease;
        }
        .footer-link:hover {
            color: #4361ee;
        }
        .footer-bottom {
            text-align: center;
            padding-top: 20px;
            border-top: 1px solid #444;
            font-size: 13px;
            color: #aaa;
        }

        /* 移动端适配（768px以下） */
        @media (max-width: 768px) {
            /* 顶部导航 */
            .navbar {
                padding: 0 20px;
            }
            .nav-menu {
                position: fixed;
                top: 70px;
                left: -100%;
                flex-direction: column;
                background-color: #fff;
                width: 100%;
                height: calc(100vh - 70px);
                padding: 30px 20px;
                transition: left 0.3s ease;
                box-shadow: 0 5px 10px rgba(0, 0, 0, 0.05);
            }
            .nav-menu.active {
                left: 0;
            }
            .nav-item {
                margin: 0 0 20px 0;
            }
            .nav-link {
                font-size: 16px;
            }
            .menu-btn {
                display: block;
            }

            /* 分类栏 */
            .category-bar {
                padding: 15px 20px;
            }
            /* 文件列表 */
            .file-section {
                padding: 20px;
            }
            .list-header {
                display: none; /* 隐藏表头 */
            }
            .file-item {
                grid-template-columns: 1fr;
                gap: 8px;
                padding: 20px;
            }
            .file-meta {
                display: inline-block;
                margin-right: 15px;
                font-size: 13px;
            }
            .download-btn {
                width: 100%;
                padding: 10px 0;
                margin-top: 10px;
            }

            /* 页脚 */
            .footer-container {
                grid-template-columns: 1fr;
                gap: 30px;
            }
        }
    </style>
    <!-- 引入图标库 -->
    <link rel="stylesheet" href="https://cdn.bootcdn.net/ajax/libs/font-awesome/6.4.0/css/all.min.css">
</head>
<body>
    <!-- 顶部导航栏 -->
    <nav class="navbar">
        <div class="nav-container">
            <a href="index.php" class="logo">
                <i class="fa-solid fa-cloud-arrow-down"></i> 文件下载中心
            </a>
            <button class="menu-btn" id="menuBtn">
                <i class="fa-solid fa-bars"></i>
            </button>
            <ul class="nav-menu" id="navMenu">
                <li class="nav-item">
                    <a href="index.php" class="nav-link active">首页</a>
                </li>
                <li class="nav-item">
                    <a href="#" class="nav-link">资源分类</a>
                </li>
                <li class="nav-item">
                    <a href="#" class="nav-link">使用帮助</a>
                </li>
                <li class="nav-item">
                    <a href="#" class="nav-link">联系我们</a>
                </li>
            </ul>
        </div>
    </nav>

    <!-- 主体内容区 -->
    <main class="main">
        <div class="container">
            <!-- 分类筛选栏 -->
            <div class="category-bar">
                <div class="category-title">文件分类</div>
                <div class="category-list">
                    <a href="index.php?category=全部" class="category-item <?php echo $activeCategory === '全部' ? 'active' : ''; ?>">
                        全部文件 (<?php echo count($allFiles); ?>)
                    </a>
                    <?php foreach ($fileCategories as $category => $exts): ?>
                        <?php $count = count($categoryFiles[$category]); ?>
                        <a href="index.php?category=<?php echo urlencode($category); ?>" class="category-item <?php echo $activeCategory === $category ? 'active' : ''; ?>">
                            <?php echo $category; ?> (<?php echo $count; ?>)
                        </a>
                    <?php endforeach; ?>
                </div>
            </div>

            <!-- 文件列表区域 -->
            <div class="file-section">
                <div class="section-header">
                    <h2 class="section-title"><?php echo $activeCategory; ?> 文件列表</h2>
                    <div class="file-count">共 <?php echo count($displayFiles); ?> 个文件</div>
                </div>

                <?php if (empty($displayFiles)): ?>
                    <!-- 无文件提示 -->
                    <div class="no-files">
                        <i class="fa-solid fa-file-circle-xmark"></i>
                        该分类下暂无可下载文件
                    </div>
                <?php else: ?>
                    <!-- 列表表头 -->
                    <div class="list-header">
                        <div>文件名称</div>
                        <div>文件大小</div>
                        <div>修改时间</div>
                        <div>操作</div>
                    </div>
                    <!-- 文件列表 -->
                    <ul class="file-list">
                        <?php foreach ($displayFiles as $file): ?>
                            <li class="file-item">
                                <div class="file-name">
                                    <i class="fa-solid <?php 
                                        // 根据文件分类显示不同图标
                                        switch ($file['category']) {
                                            case '文档类': echo 'fa-file-lines'; break;
                                            case '压缩包': echo 'fa-file-zipper'; break;
                                            case '图片类': echo 'fa-file-image'; break;
                                            case '视频类': echo 'fa-file-video'; break;
                                            case '音频类': echo 'fa-file-audio'; break;
                                            default: echo 'fa-file';
                                        }
                                    ?>"></i>
                                    <?php echo htmlspecialchars($file['name']); // 防XSS攻击 ?>
                                </div>
                                <div class="file-meta">
                                    <?php echo formatFileSize($file['size']); // 格式化文件大小 ?>
                                </div>
                                <div class="file-meta">
                                    <?php echo formatTime($file['mtime']); // 格式化修改时间 ?>
                                </div>
                                <a href="?download=<?php echo urlencode($file['name']); ?>" class="download-btn">
                                    <i class="fa-solid fa-download"></i> 下载
                                </a>
                            </li>
                        <?php endforeach; ?>
                    </ul>
                <?php endif; ?>
            </div>
        </div>
    </main>

    <!-- 底部导航栏（页脚） -->
    <footer class="footer">
        <div class="footer-container">
            <div class="footer-col">
                <h3>关于我们</h3>
                <a href="#" class="footer-link">平台介绍</a>
                <a href="#" class="footer-link">资源说明</a>
                <a href="#" class="footer-link">隐私政策</a>
            </div>
            <div class="footer-col">
                <h3>资源分类</h3>
                <a href="index.php?category=文档类" class="footer-link">文档类资源</a>
                <a href="index.php?category=压缩包" class="footer-link">压缩包资源</a>
                <a href="index.php?category=图片类" class="footer-link">图片类资源</a>
                <a href="index.php?category=视频类" class="footer-link">视频类资源</a>
                <a href="index.php?category=音频类" class="footer-link">音频类资源</a>
            </div>
            <div class="footer-col">
                <h3>联系我们</h3>
                <a href="#" class="footer-link"><i class="fa-solid fa-envelope"></i> 邮箱咨询</a>
                <a href="#" class="footer-link"><i class="fa-solid fa-phone"></i> 电话联系</a>
                <a href="#" class="footer-link"><i class="fa-solid fa-qq"></i> QQ交流群</a>
            </div>
        </div>
        <div class="footer-bottom">
            © <?php echo date('Y'); ?> 文件下载中心 版权所有 | 仅供学习交流使用，请勿用于商业用途
        </div>
    </footer>

    <!-- 移动端菜单交互脚本 -->
    <script>
        const menuBtn = document.getElementById('menuBtn');
        const navMenu = document.getElementById('navMenu');
        
        menuBtn.addEventListener('click', () => {
            navMenu.classList.toggle('active');
            // 切换菜单图标（汉堡→叉号）
            const icon = menuBtn.querySelector('i');
            if (navMenu.classList.contains('active')) {
                icon.classList.remove('fa-bars');
                icon.classList.add('fa-times');
            } else {
                icon.classList.remove('fa-times');
                icon.classList.add('fa-bars');
            }
        });

        // 点击导航链接后关闭移动端菜单
        const navLinks = document.querySelectorAll('.nav-link');
        navLinks.forEach(link => {
            link.addEventListener('click', () => {
                if (navMenu.classList.contains('active')) {
                    navMenu.classList.remove('active');
                    menuBtn.querySelector('i').classList.remove('fa-times');
                    menuBtn.querySelector('i').classList.add('fa-bars');
                }
            });
        });
    </script>
</body>
</html>
